<?php
class AdminActivityController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Obtiene todas las actividades económicas pendientes de verificación.
     */
    public function getPendingActivities() {
        $sql = "SELECT 
                    e.id, e.activity_code, e.description, e.created_at,
                    u.document_number, u.company_name, u.first_name, u.last_name
                FROM economic_activities e
                JOIN users u ON e.user_id = u.id
                WHERE e.status = 'pending_verification'
                ORDER BY e.created_at ASC";
        
        $stmt = $this->db->query($sql);
        return $stmt->fetchAll();
    }

    /**
     * Obtiene los detalles completos de una actividad, incluyendo su propietario y documentos.
     * @param int $activityId El ID de la actividad.
     * @return array|false Los detalles de la actividad o false si no se encuentra.
     */
    public function getActivityDetails($activityId) {
        $sql = "SELECT 
                    e.*,
                    u.document_number, u.company_name, u.first_name, u.last_name,
                    a.file_path, a.file_name
                FROM economic_activities e
                JOIN users u ON e.user_id = u.id
                LEFT JOIN attachments a ON e.id = a.entity_id AND a.entity_type = 'economic_activity'
                WHERE e.id = ?";
        
        $stmt = $this->db->query($sql, [$activityId]);
        return $stmt->fetch();
    }

    /**
     * Aprueba el registro de una actividad económica.
     * @param int $activityId El ID de la actividad a aprobar.
     * @return array Resultado de la operación.
     */
    public function approveActivity($activityId) {
        $activity = $this->getActivityDetails($activityId);
        if (!$activity || $activity['status'] !== 'pending_verification') {
            return ['success' => false, 'message' => 'La actividad no es válida o ya fue procesada.'];
        }

        $sql = "UPDATE economic_activities SET status = 'active', verified_at = NOW() WHERE id = ?";
        $stmt = $this->db->query($sql, [$activityId]);

        if ($stmt) {
            logActivity('activity_approved', 'economic_activities', $activityId);
            $message = "La actividad económica '" . $activity['description'] . "' ha sido aprobada. Ya puede declarar sus ingresos.";
            create_notification($activity['user_id'], $message, 'mis-actividades.php');
            return ['success' => true, 'message' => 'Actividad económica aprobada exitosamente.'];
        }
        return ['success' => false, 'message' => 'Error al aprobar la actividad.'];
    }

    /**
     * Rechaza el registro de una actividad económica.
     * @param int $activityId El ID de la actividad a rechazar.
     * @param string $reason El motivo del rechazo.
     * @return array Resultado de la operación.
     */
    public function rejectActivity($activityId, $reason) {
        $activity = $this->getActivityDetails($activityId);
        if (!$activity) {
            return ['success' => false, 'message' => 'Actividad no encontrada.'];
        }

        $sql = "UPDATE economic_activities SET status = 'rejected', rejection_reason = ? WHERE id = ?";
        $stmt = $this->db->query($sql, [$reason, $activityId]);
        
        if ($stmt) {
            logActivity('activity_rejected', 'economic_activities', $activityId);
            $message = "Su solicitud para la actividad '" . $activity['description'] . "' fue rechazada. Motivo: " . $reason;
            create_notification($activity['user_id'], $message, 'mis-actividades.php');
            return ['success' => true, 'message' => 'Actividad rechazada exitosamente.'];
        }
        return ['success' => false, 'message' => 'Error al rechazar la actividad.'];
    }
}
?>