<?php
class AdminAdvertisementController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    public function getPendingAdvertisements() {
        $sql = "SELECT 
                    a.id, a.ad_type, a.description, a.created_at,
                    u.document_number, u.company_name, u.first_name, u.last_name
                FROM advertisements a
                JOIN users u ON a.user_id = u.id
                WHERE a.status = 'pending_verification'
                ORDER BY a.created_at ASC";
        
        $stmt = $this->db->query($sql);
        return $stmt->fetchAll();
    }

    public function getAdvertisementDetails($adId) {
        $sql = "SELECT 
                    ad.*,
                    u.document_number, u.company_name, u.first_name, u.last_name,
                    att.file_path, att.file_name
                FROM advertisements ad
                JOIN users u ON ad.user_id = u.id
                LEFT JOIN attachments att ON ad.id = att.entity_id AND att.entity_type = 'advertisement'
                WHERE ad.id = ?";
        
        $stmt = $this->db->query($sql, [$adId]);
        return $stmt->fetch();
    }

    public function approveAdvertisement($adId) {
        $ad = $this->getAdvertisementDetails($adId);
        if (!$ad || $ad['status'] !== 'pending_verification') {
            return ['success' => false, 'message' => 'El anuncio no es válido o ya fue procesado.'];
        }
        
        // --- INICIO DE LA LÓGICA PARA GENERAR IMPUESTO ---
        $conceptId = getSystemConfig('advertisement_tax_concept_id');
        if (empty($conceptId)) {
            return ['success' => false, 'message' => 'No se ha configurado un concepto tributario para publicidad en los ajustes del sistema.'];
        }
        $concept = $this->db->query("SELECT * FROM tax_concepts WHERE id = ?", [$conceptId])->fetch();
        if (!$concept) {
            return ['success' => false, 'message' => 'El concepto tributario para publicidad configurado no fue encontrado.'];
        }

        // Lógica de cálculo simple (ejemplo)
        $amount = $concept['base_amount'] ?? 250.00;
        if ($ad['ad_type'] === 'Valla Publicitaria') {
            $amount *= 2;
        }

        $defaultCurrency = $this->db->query("SELECT id FROM currencies WHERE is_default = 1")->fetch();
        if (!$defaultCurrency) {
            return ['success' => false, 'message' => 'No hay una moneda predeterminada configurada.'];
        }

        $taxController = new TaxController();
        $taxData = [
            'user_id' => $ad['user_id'],
            'tax_concept_id' => $conceptId,
            'description' => "Impuesto de Publicidad: " . $ad['description'],
            'amount' => round($amount, 2),
            'currency_id' => $defaultCurrency['id'],
            'due_date' => date('Y-m-d', strtotime('+30 days'))
        ];

        if (!$taxController->createTax($taxData)) {
            return ['success' => false, 'message' => 'Error al generar la deuda del impuesto para el anuncio.'];
        }
        // --- FIN DE LA LÓGICA PARA GENERAR IMPUESTO ---

        $sql = "UPDATE advertisements SET status = 'verified' WHERE id = ?";
        $stmt = $this->db->query($sql, [$adId]);

        if ($stmt) {
            logActivity('ad_approved', 'advertisements', $adId);
            $message = "Su solicitud para el anuncio publicitario '" . $ad['description'] . "' ha sido aprobada y el impuesto correspondiente ha sido generado.";
            create_notification($ad['user_id'], $message, 'mis-anuncios.php');
            return ['success' => true, 'message' => 'Anuncio aprobado e impuesto generado exitosamente.'];
        }
        return ['success' => false, 'message' => 'Error al aprobar el anuncio.'];
    }

    public function rejectAdvertisement($adId, $reason) {
        $ad = $this->getAdvertisementDetails($adId);
        if (!$ad) return ['success' => false, 'message' => 'Anuncio no encontrado.'];

        $sql = "UPDATE advertisements SET status = 'rejected', rejection_reason = ? WHERE id = ?";
        $stmt = $this->db->query($sql, [$reason, $adId]);
        
        if ($stmt) {
            logActivity('ad_rejected', 'advertisements', $adId);
            $message = "Su solicitud para el anuncio '" . $ad['description'] . "' fue rechazada. Motivo: " . $reason;
            create_notification($ad['user_id'], $message, 'mis-anuncios.php');
            return ['success' => true, 'message' => 'Anuncio rechazado exitosamente.'];
        }
        return ['success' => false, 'message' => 'Error al rechazar el anuncio.'];
    }
}
?>