<?php
class AdminFiscalizationController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Obtiene todas las auditorías fiscales realizadas.
     */
    public function getAllAudits() {
        $sql = "SELECT 
                    fa.*,
                    contrib.document_number as contrib_document,
                    contrib.company_name, contrib.first_name, contrib.last_name,
                    fiscal.first_name as fiscal_first_name, fiscal.last_name as fiscal_last_name
                FROM fiscal_audits fa
                JOIN users contrib ON fa.user_id = contrib.id
                JOIN users fiscal ON fa.fiscal_id = fiscal.id
                ORDER BY fa.visit_date DESC";
        return $this->db->query($sql)->fetchAll();
    }

    /**
     * Obtiene todos los tipos de infracciones.
     */
    public function getInfractionTypes() {
        return $this->db->query("SELECT * FROM infraction_types ORDER BY code")->fetchAll();
    }

    /**
     * Crea una nueva auditoría y le asocia las infracciones, generando las multas.
     */
    public function createAudit($data) {
        $fiscalId = $_SESSION['user_id'];
        $userId = $data['user_id'];
        $visitDate = $data['visit_date'];
        $notes = $data['notes'];
        $infractionIds = $data['infractions'] ?? [];

        // 1. Crear el registro de la auditoría
        $sqlAudit = "INSERT INTO fiscal_audits (user_id, fiscal_id, visit_date, notes) VALUES (?, ?, ?, ?)";
        $stmtAudit = $this->db->query($sqlAudit, [$userId, $fiscalId, $visitDate, $notes]);
        if (!$stmtAudit) {
            return ['success' => false, 'message' => 'Error al registrar la visita de fiscalización.'];
        }
        $auditId = $this->db->lastInsertId();

        // 2. Procesar infracciones y generar multas si las hay
        if (!empty($infractionIds)) {
            $taxController = new TaxController();
            $defaultCurrency = $this->db->query("SELECT id FROM currencies WHERE is_default = 1")->fetch();

            foreach ($infractionIds as $infractionId) {
                $infractionType = $this->db->query("SELECT * FROM infraction_types WHERE id = ?", [$infractionId])->fetch();
                if ($infractionType) {
                    // Crear la multa como un nuevo impuesto
                    $taxData = [
                        'user_id' => $userId,
                        'tax_concept_id' => getSystemConfig('fine_tax_concept_id'), // Necesitaremos configurar esto
                        'property_id' => null,
                        'description' => 'Multa por Infracción: ' . $infractionType['description'],
                        'amount' => $infractionType['fine_amount'],
                        'currency_id' => $defaultCurrency['id'],
                        'due_date' => date('Y-m-d', strtotime('+15 days'))
                    ];
                    $taxId = $taxController->createTax($taxData);

                    // Asociar la infracción con la auditoría y la multa generada
                    $this->db->query(
                        "INSERT INTO fiscal_audit_infractions (audit_id, infraction_type_id, tax_id) VALUES (?, ?, ?)",
                        [$auditId, $infractionId, $taxId]
                    );
                }
            }
            // Actualizar el estado de la auditoría
            $this->db->query("UPDATE fiscal_audits SET status = 'pending_fine' WHERE id = ?", [$auditId]);
        }

        logActivity('fiscal_audit_created', 'fiscal_audits', $auditId);
        return ['success' => true, 'message' => 'Fiscalización registrada y multas generadas exitosamente.'];
    }
}
?>