<?php
class AdminLicenseController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Obtiene todas las solicitudes de licencia pendientes.
     */
    public function getPendingLicenses() {
        $sql = "SELECT 
                    l.id, l.request_date,
                    u.document_number, u.company_name, u.first_name, u.last_name,
                    e.activity_code, e.description as activity_description
                FROM economic_activity_licenses l
                JOIN users u ON l.user_id = u.id
                JOIN economic_activities e ON l.economic_activity_id = e.id
                WHERE l.status = 'solicitada'
                ORDER BY l.request_date ASC";
        
        $stmt = $this->db->query($sql);
        return $stmt->fetchAll();
    }

    /**
     * Obtiene los detalles completos de una solicitud de licencia.
     */
    public function getLicenseDetails($licenseId) {
        $sql = "SELECT 
                    l.*,
                    u.document_number, u.company_name, u.first_name, u.last_name,
                    e.activity_code, e.description as activity_description
                FROM economic_activity_licenses l
                JOIN users u ON l.user_id = u.id
                JOIN economic_activities e ON l.economic_activity_id = e.id
                WHERE l.id = ?";
        
        $stmt = $this->db->query($sql, [$licenseId]);
        return $stmt->fetch();
    }

    /**
     * Aprueba una solicitud de licencia.
     */
    public function approveLicense($licenseId) {
        $license = $this->getLicenseDetails($licenseId);
        if (!$license || $license['status'] !== 'solicitada') {
            return ['success' => false, 'message' => 'La solicitud no es válida o ya fue procesada.'];
        }

        $licenseNumber = 'LIC-' . date('Y') . '-' . str_pad($licenseId, 6, '0', STR_PAD_LEFT);
        $approvalDate = date('Y-m-d H:i:s');
        $expiryDate = date('Y-m-d', strtotime('+1 year'));

        $sql = "UPDATE economic_activity_licenses 
                SET status = 'aprobada', 
                    license_number = ?,
                    approval_date = ?,
                    expiry_date = ?,
                    reviewed_by = ?
                WHERE id = ?";
        
        $stmt = $this->db->query($sql, [$licenseNumber, $approvalDate, $expiryDate, $_SESSION['user_id'], $licenseId]);

        if ($stmt) {
            logActivity('license_approved', 'economic_activity_licenses', $licenseId);
            $message = "Su licencia para la actividad '" . $license['activity_description'] . "' ha sido aprobada. Número de licencia: " . $licenseNumber;
            create_notification($license['user_id'], $message, 'licencias.php');
            return ['success' => true, 'message' => 'Licencia aprobada y generada exitosamente.'];
        }
        return ['success' => false, 'message' => 'Error al aprobar la licencia.'];
    }

    /**
     * Rechaza una solicitud de licencia.
     */
    public function rejectLicense($licenseId, $reason) {
        $license = $this->getLicenseDetails($licenseId);
        if (!$license) {
            return ['success' => false, 'message' => 'Solicitud no encontrada.'];
        }

        $sql = "UPDATE economic_activity_licenses 
                SET status = 'rechazada', rejection_reason = ?, reviewed_by = ?
                WHERE id = ?";
        $stmt = $this->db->query($sql, [$reason, $_SESSION['user_id'], $licenseId]);
        
        if ($stmt) {
            logActivity('license_rejected', 'economic_activity_licenses', $licenseId);
            $message = "Su solicitud de licencia para la actividad '" . $license['activity_description'] . "' fue rechazada. Motivo: " . $reason;
            create_notification($license['user_id'], $message, 'licencias.php');
            return ['success' => true, 'message' => 'Solicitud rechazada exitosamente.'];
        }
        return ['success' => false, 'message' => 'Error al rechazar la solicitud.'];
    }
}
?>