<?php
class AdminPaymentController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Obtiene todos los pagos que están pendientes de verificación.
     * @return array Lista de pagos pendientes con detalles del usuario e impuesto.
     */
    public function getPendingPayments() {
        $sql = "SELECT 
                    p.id, p.amount, p.reference_number, p.bank_name, p.payment_method, p.created_at, p.receipt_url,
                    u.document_number, u.first_name, u.last_name, u.company_name,
                    tc.name as tax_concept,
                    c.symbol as currency_symbol
                FROM payments p
                JOIN users u ON p.user_id = u.id
                JOIN taxes t ON p.tax_id = t.id
                JOIN tax_concepts tc ON t.tax_concept_id = tc.id
                JOIN currencies c ON p.currency_id = c.id
                WHERE p.status = 'pending'
                ORDER BY p.created_at ASC";
        
        $stmt = $this->db->query($sql);
        return $stmt->fetchAll();
    }

    /**
     * Aprueba un pago.
     * @param int $paymentId El ID del pago a aprobar.
     * @return bool True si la operación fue exitosa, false si no.
     */
    public function approvePayment($paymentId) {
        // 1. Obtener el tax_id asociado al pago
        $stmt = $this->db->query("SELECT tax_id FROM payments WHERE id = ? AND status = 'pending'", [$paymentId]);
        $payment = $stmt->fetch();
        if (!$payment) return false;

        // 2. Actualizar el estado del pago a 'verified'
        $this->db->query("UPDATE payments SET status = 'verified', verified_at = NOW() WHERE id = ?", [$paymentId]);

        // 3. Actualizar el estado del impuesto a 'paid'
        $this->db->query("UPDATE taxes SET status = 'paid', payment_date = NOW() WHERE id = ?", [$payment['tax_id']]);

        logActivity('payment_approved', 'payments', $paymentId);
        return true;
    }

    /**
     * Rechaza un pago.
     * @param int $paymentId El ID del pago a rechazar.
     * @return bool True si la operación fue exitosa, false si no.
     */
    public function rejectPayment($paymentId) {
        // 1. Obtener el tax_id asociado al pago
        $stmt = $this->db->query("SELECT tax_id FROM payments WHERE id = ? AND status = 'pending'", [$paymentId]);
        $payment = $stmt->fetch();
        if (!$payment) return false;

        // 2. Actualizar el estado del pago a 'rejected'
        $this->db->query("UPDATE payments SET status = 'rejected', verified_at = NOW() WHERE id = ?", [$paymentId]);

        // 3. Devolver el estado del impuesto a 'pending' para que el usuario pueda intentarlo de nuevo
        $this->db->query("UPDATE taxes SET status = 'pending' WHERE id = ?", [$payment['tax_id']]);

        logActivity('payment_rejected', 'payments', $paymentId);
        return true;
    }
}
?>