<?php
class AdminPermitController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Obtiene todas las solicitudes de permisos pendientes de un departamento específico.
     */
    public function getPendingPermitsByDepartment($department) {
        $sql = "SELECT 
                    pa.id, pa.application_date,
                    pt.name as permit_name,
                    u.document_number, u.company_name, u.first_name, u.last_name
                FROM permit_applications pa
                JOIN users u ON pa.user_id = u.id
                JOIN permit_types pt ON pa.permit_type_id = pt.id
                WHERE pa.status = 'solicitado' AND pt.issuing_department = ?
                ORDER BY pa.application_date ASC";
        
        $stmt = $this->db->query($sql, [$department]);
        return $stmt->fetchAll();
    }

    /**
     * Obtiene los detalles completos de una solicitud de permiso.
     */
    public function getPermitDetails($permitId) {
        $sql = "SELECT 
                    pa.*,
                    pt.name as permit_name,
                    u.document_number, u.company_name, u.first_name, u.last_name, u.email, u.phone,
                    prop.cadastral_code, prop.address as property_address,
                    att.file_path, att.file_name
                FROM permit_applications pa
                JOIN users u ON pa.user_id = u.id
                JOIN permit_types pt ON pa.permit_type_id = pt.id
                LEFT JOIN properties prop ON pa.related_entity_id = prop.id AND pa.related_entity_type = 'property'
                LEFT JOIN attachments att ON pa.id = att.entity_id AND att.entity_type = 'permit_application'
                WHERE pa.id = ?";
        
        $stmt = $this->db->query($sql, [$permitId]);
        return $stmt->fetch();
    }
    
    /**
     * Aprueba una solicitud de permiso.
     */
    public function approvePermit($permitId) {
        $permit = $this->getPermitDetails($permitId);
        if (!$permit || $permit['status'] !== 'solicitado') {
            return ['success' => false, 'message' => 'La solicitud no es válida o ya fue procesada.'];
        }

        $permitNumber = 'PER-' . date('Y') . '-' . str_pad($permitId, 6, '0', STR_PAD_LEFT);
        $approvalDate = date('Y-m-d H:i:s');
        $expiryDate = date('Y-m-d', strtotime('+1 year'));

        $sql = "UPDATE permit_applications 
                SET status = 'aprobado', 
                    permit_number = ?,
                    approval_date = ?,
                    expiry_date = ?,
                    reviewed_by = ?
                WHERE id = ?";
        
        $stmt = $this->db->query($sql, [$permitNumber, $approvalDate, $expiryDate, $_SESSION['user_id'], $permitId]);

        if ($stmt) {
            logActivity('permit_approved', 'permit_applications', $permitId);
            $message = "Su solicitud para '" . $permit['permit_name'] . "' ha sido aprobada. Número de permiso: " . $permitNumber;
            create_notification($permit['user_id'], $message, 'mis-permisos.php');
            return ['success' => true, 'message' => 'Permiso aprobado y generado exitosamente.'];
        }
        return ['success' => false, 'message' => 'Error al aprobar el permiso.'];
    }

    /**
     * Rechaza una solicitud de permiso.
     */
    public function rejectPermit($permitId, $reason) {
        $permit = $this->getPermitDetails($permitId);
        if (!$permit) return ['success' => false, 'message' => 'Solicitud no encontrada.'];

        $sql = "UPDATE permit_applications SET status = 'rechazada', rejection_reason = ?, reviewed_by = ? WHERE id = ?";
        $stmt = $this->db->query($sql, [$reason, $_SESSION['user_id'], $permitId]);
        
        if ($stmt) {
            logActivity('permit_rejected', 'permit_applications', $permitId);
            $message = "Su solicitud para '" . $permit['permit_name'] . "' fue rechazada. Motivo: " . $reason;
            create_notification($permit['user_id'], $message, 'mis-permisos.php');
            return ['success' => true, 'message' => 'Solicitud rechazada exitosamente.'];
        }
        return ['success' => false, 'message' => 'Error al rechazar la solicitud.'];
    }
}
?>