<?php
class AdminPropertyController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    public function getPendingProperties() {
        $sql = "SELECT 
                    p.id, p.cadastral_code, p.address, p.created_at,
                    u.document_number, u.company_name, u.first_name, u.last_name
                FROM properties p
                JOIN users u ON p.user_id = u.id
                WHERE p.status = 'pending_verification'
                ORDER BY p.created_at ASC";
        
        $stmt = $this->db->query($sql);
        return $stmt->fetchAll();
    }

    public function getPropertyDetails($propertyId) {
        $sql = "SELECT 
                    p.*,
                    u.document_number, u.company_name, u.first_name, u.last_name,
                    a.file_path, a.file_name
                FROM properties p
                JOIN users u ON p.user_id = u.id
                LEFT JOIN attachments a ON p.id = a.entity_id AND a.entity_type = 'property'
                WHERE p.id = ?";
        
        $stmt = $this->db->query($sql, [$propertyId]);
        return $stmt->fetch();
    }

    public function approveProperty($propertyId) {
        $property = $this->getPropertyDetails($propertyId);
        if (!$property || $property['status'] !== 'pending_verification') {
            return ['success' => false, 'message' => 'El inmueble no es válido o ya fue procesado.'];
        }

        // --- INICIO DE LA LÓGICA PARA GENERAR IMPUESTO ---

        // 1. Obtener el concepto de impuesto para inmuebles desde la configuración del sistema.
        //    (Crearemos esta configuración más adelante)
        $conceptId = getSystemConfig('property_tax_concept_id'); 
        if (empty($conceptId)) {
            return ['success' => false, 'message' => 'No se ha configurado un concepto tributario para inmuebles en los ajustes del sistema.'];
        }
        $concept = $this->db->query("SELECT * FROM tax_concepts WHERE id = ?", [$conceptId])->fetch();
        if (!$concept) {
            return ['success' => false, 'message' => 'El concepto tributario para inmuebles configurado no fue encontrado.'];
        }

        // 2. Calcular el monto del impuesto (ejemplo: basado en m² y un monto base)
        $amount = ($property['area_m2'] * ($concept['aliquota'] ?? 0.5)) + ($concept['base_amount'] ?? 50);

        $defaultCurrency = $this->db->query("SELECT id FROM currencies WHERE is_default = 1")->fetch();
        if (!$defaultCurrency) {
            return ['success' => false, 'message' => 'No hay una moneda predeterminada configurada.'];
        }
        
        // 3. Crear la deuda en la tabla 'taxes'
        $taxController = new TaxController();
        $taxData = [
            'user_id' => $property['user_id'],
            'tax_concept_id' => $conceptId,
            'property_id' => $propertyId, // Asociamos el impuesto al inmueble
            'description' => "Impuesto sobre Inmueble Urbano - Catastro: " . $property['cadastral_code'],
            'amount' => round($amount, 2),
            'currency_id' => $defaultCurrency['id'],
            'due_date' => date('Y-m-d', strtotime('+30 days'))
        ];

        if (!$taxController->createTax($taxData)) {
            return ['success' => false, 'message' => 'Error al generar la deuda del impuesto para el inmueble.'];
        }

        // --- FIN DE LA LÓGICA PARA GENERAR IMPUESTO ---

        // 4. Actualizar el estado del inmueble a 'verificado'
        $sql = "UPDATE properties SET status = 'verified' WHERE id = ?";
        $stmt = $this->db->query($sql, [$propertyId]);

        if ($stmt) {
            logActivity('property_approved', 'properties', $propertyId);
            $message = "El registro del inmueble con Cédula Catastral " . $property['cadastral_code'] . " ha sido aprobado y su impuesto ha sido generado.";
            create_notification($property['user_id'], $message, 'mis-inmuebles.php');
            return ['success' => true, 'message' => 'Inmueble aprobado e impuesto generado exitosamente.'];
        }
        return ['success' => false, 'message' => 'Error al aprobar el inmueble.'];
    }

    public function rejectProperty($propertyId, $reason) {
        $property = $this->getPropertyDetails($propertyId);
        if (!$property) {
            return ['success' => false, 'message' => 'Inmueble no encontrado.'];
        }

        $sql = "UPDATE properties SET status = 'rejected', rejection_reason = ? WHERE id = ?";
        $stmt = $this->db->query($sql, [$reason, $propertyId]);
        
        if ($stmt) {
            logActivity('property_rejected', 'properties', $propertyId);
            $message = "Su solicitud para el inmueble " . $property['cadastral_code'] . " fue rechazada. Motivo: " . $reason;
            create_notification($property['user_id'], $message, 'mis-inmuebles.php');
            return ['success' => true, 'message' => 'Inmueble rechazado exitosamente.'];
        }
        return ['success' => false, 'message' => 'Error al rechazar el inmueble.'];
    }
}
?>