<?php
class AdminReportController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Obtiene todos los reportes ciudadanos, uniendo la información del usuario.
     */
    public function getAllReports() {
        $sql = "SELECT 
                    cr.*,
                    u.document_number, u.first_name, u.last_name, u.company_name
                FROM citizen_reports cr
                JOIN users u ON cr.user_id = u.id
                ORDER BY 
                    CASE cr.status
                        WHEN 'open' THEN 1
                        WHEN 'in_progress' THEN 2
                        WHEN 'resolved' THEN 3
                        WHEN 'closed' THEN 4
                    END,
                    cr.created_at DESC";
        
        $stmt = $this->db->query($sql);
        return $stmt->fetchAll();
    }

    /**
     * Obtiene los detalles completos de un reporte, incluyendo su creador y adjuntos.
     * @param int $reportId El ID del reporte.
     * @return array|false Los detalles del reporte o false si no se encuentra.
     */
    public function getReportDetails($reportId) {
        $sql = "SELECT 
                    cr.*,
                    u.document_number, u.first_name, u.last_name, u.company_name,
                    a.file_path, a.file_name
                FROM citizen_reports cr
                JOIN users u ON cr.user_id = u.id
                LEFT JOIN attachments a ON cr.id = a.entity_id AND a.entity_type = 'citizen_report'
                WHERE cr.id = ?";
        
        $stmt = $this->db->query($sql, [$reportId]);
        return $stmt->fetch();
    }

    /**
     * Actualiza el estado de un reporte ciudadano.
     * @param int $reportId El ID del reporte.
     * @param string $newStatus El nuevo estado.
     * @return array Resultado de la operación.
     */
    public function updateReportStatus($reportId, $newStatus) {
        $report = $this->getReportDetails($reportId);
        if (!$report) {
            return ['success' => false, 'message' => 'Reporte no encontrado.'];
        }

        // Marcar la fecha de resolución si el estado es 'resolved'
        $resolvedAt = ($newStatus === 'resolved') ? date('Y-m-d H:i:s') : null;

        $sql = "UPDATE citizen_reports SET status = ?, resolved_at = ? WHERE id = ?";
        $stmt = $this->db->query($sql, [$newStatus, $resolvedAt, $reportId]);

        if ($stmt) {
            logActivity('report_status_updated', 'citizen_reports', $reportId, ['status' => $report['status']], ['status' => $newStatus]);
            $message = "El estado del reporte '" . $report['title'] . "' ha sido actualizado a: " . ucfirst(str_replace('_', ' ', $newStatus));
            create_notification($report['user_id'], $message, 'mis-reportes.php');
            return ['success' => true, 'message' => 'Estado del reporte actualizado exitosamente.'];
        }
        return ['success' => false, 'message' => 'Error al actualizar el estado del reporte.'];
    }
}
?>