<?php
class AdminVehicleController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Obtiene todos los vehículos pendientes de verificación.
     * @return array Lista de vehículos con detalles del propietario.
     */
    public function getPendingVehicles() {
        $sql = "SELECT 
                    v.id, v.plate_number, v.brand, v.model, v.year, v.created_at,
                    u.document_number, u.first_name, u.last_name, u.company_name
                FROM vehicles v
                JOIN users u ON v.user_id = u.id
                WHERE v.status = 'pending_verification'
                ORDER BY v.created_at ASC";
        
        $stmt = $this->db->query($sql);
        return $stmt->fetchAll();
    }

    /**
     * Obtiene los detalles completos de un vehículo, incluyendo su propietario y documentos.
     * @param int $vehicleId El ID del vehículo.
     * @return array|false Los detalles del vehículo o false si no se encuentra.
     */
    public function getVehicleDetails($vehicleId) {
        $sql = "SELECT 
                    v.*,
                    u.document_number, u.first_name, u.last_name, u.company_name,
                    a.file_path, a.file_name
                FROM vehicles v
                JOIN users u ON v.user_id = u.id
                LEFT JOIN attachments a ON v.id = a.entity_id AND a.entity_type = 'vehicle'
                WHERE v.id = ?";
        
        $stmt = $this->db->query($sql, [$vehicleId]);
        return $stmt->fetch();
    }

    /**
     * Aprueba la declaración de un vehículo y genera el impuesto correspondiente.
     * @param int $vehicleId El ID del vehículo a aprobar.
     * @return array Resultado de la operación.
     */
    public function approveVehicle($vehicleId) {
        $vehicle = $this->getVehicleDetails($vehicleId);
        if (!$vehicle || $vehicle['status'] !== 'pending_verification') {
            return ['success' => false, 'message' => 'El vehículo no es válido o ya fue procesado.'];
        }

        $conceptId = getSystemConfig('vehicle_tax_concept_id');
        if (empty($conceptId)) {
            return ['success' => false, 'message' => 'No se ha configurado un concepto tributario para vehículos en los ajustes del sistema.'];
        }
        $concept = $this->db->query("SELECT * FROM tax_concepts WHERE id = ?", [$conceptId])->fetch();
        if (!$concept) {
            return ['success' => false, 'message' => 'El concepto tributario configurado (ID: '.$conceptId.') no fue encontrado.'];
        }

        $amount = $concept['base_amount'] ?? 100.00;
        if ($vehicle['year'] < 2010) $amount *= 0.8;
        if ($vehicle['vehicle_type'] === 'carga') $amount *= 1.5;

        $defaultCurrency = $this->db->query("SELECT id FROM currencies WHERE is_default = 1")->fetch();
        if (!$defaultCurrency) {
            return ['success' => false, 'message' => 'No hay una moneda predeterminada configurada.'];
        }

        $sqlTax = "INSERT INTO taxes (user_id, tax_concept_id, description, amount, currency_id, due_date, status) VALUES (?, ?, ?, ?, ?, ?, 'pending')";
        $description = "Impuesto sobre vehículo placa: " . $vehicle['plate_number'];
        $dueDate = date('Y-m-d', strtotime('+30 days'));
        $params = [$vehicle['user_id'], $conceptId, $description, round($amount, 2), $defaultCurrency['id'], $dueDate];
        $stmtTax = $this->db->query($sqlTax, $params);

        if (!$stmtTax) {
            return ['success' => false, 'message' => 'Error al generar la deuda del impuesto.'];
        }

        $this->db->query("UPDATE vehicles SET status = 'verified' WHERE id = ?", [$vehicleId]);
        logActivity('vehicle_approved', 'vehicles', $vehicleId);
        
        $message = "Su vehículo placa " . $vehicle['plate_number'] . " ha sido aprobado. Ya puede proceder a pagar el impuesto correspondiente.";
        create_notification($vehicle['user_id'], $message, 'mis-impuestos.php');
        
        return ['success' => true, 'message' => 'Vehículo aprobado exitosamente y deuda generada.'];
    }

    /**
     * Rechaza la declaración de un vehículo.
     * @param int $vehicleId El ID del vehículo a rechazar.
     * @param string $reason El motivo del rechazo.
     * @return array Resultado de la operación.
     */
    public function rejectVehicle($vehicleId, $reason) {
        $vehicle = $this->getVehicleDetails($vehicleId);
        if (!$vehicle) {
            return ['success' => false, 'message' => 'Vehículo no encontrado.'];
        }

        $sql = "UPDATE vehicles SET status = 'rejected', rejection_reason = ? WHERE id = ?";
        $stmt = $this->db->query($sql, [$reason, $vehicleId]);
        
        if ($stmt) {
            logActivity('vehicle_rejected', 'vehicles', $vehicleId);
            
            $message = "Su solicitud para el vehículo placa " . $vehicle['plate_number'] . " fue rechazada. Motivo: " . $reason;
            create_notification($vehicle['user_id'], $message, 'mis-vehiculos.php');

            return ['success' => true, 'message' => 'Vehículo rechazado exitosamente.'];
        }
        return ['success' => false, 'message' => 'Error al rechazar el vehículo.'];
    }
}
?>