<?php
// includes/auth.php

class Auth {
    private $db;
    private $userPermissions = [];
    
    public function __construct() {
        $this->db = Database::getInstance();
        if ($this->isLoggedIn() && isset($_SESSION['permissions'])) {
            $this->userPermissions = $_SESSION['permissions'];
        }
    }
    
    public function register($data) {
        if (!$this->validateRegistrationData($data)) {
            return ['success' => false, 'message' => 'Datos de registro inválidos'];
        }
        if ($this->userExists($data['document_number'])) {
            return ['success' => false, 'message' => 'El usuario ya existe'];
        }
        
        $hashedPassword = password_hash($data['password'], PASSWORD_DEFAULT);
        
        try {
            $sanitizedDocumentNumber = strtoupper(preg_replace('/[^a-zA-Z0-9]/', '', $data['document_number']));
            $defaultRoleId = 99;
            $userType = in_array($data['user_type'], ['natural', 'juridica']) ? $data['user_type'] : 'natural';

            $stmt = $this->db->query(
                "INSERT INTO users (user_type, document_type, document_number, email, password, first_name, last_name, company_name, phone, address, role_id) 
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                [
                    $userType, $data['document_type'], $sanitizedDocumentNumber,
                    $data['email'], $hashedPassword, $data['first_name'] ?? null,
                    $data['last_name'] ?? null, $data['company_name'] ?? null,
                    $data['phone'] ?? null, $data['address'] ?? null, $defaultRoleId
                ]
            );
            
            if ($stmt) {
                $userId = $this->db->lastInsertId();
                logActivity('user_registered', 'users', $userId);
                return ['success' => true, 'user_id' => $userId];
            }
            return ['success' => false, 'message' => 'Error al crear usuario'];
        } catch(PDOException $e) {
            error_log("Registration error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Error interno del sistema'];
        }
    }
    
    public function login($documentNumber, $password) {
        $user = $this->getUserByDocument($documentNumber);
        if (!$user || !password_verify($password, $user['password']) || !$user['is_active']) {
            return ['success' => false, 'message' => 'Credenciales incorrectas o cuenta inactiva.'];
        }
        $this->createSession($user);
        logActivity('user_login', 'users', $user['id']);
        return ['success' => true, 'user' => $user];
    }
    
    public function logout() {
        if (isset($_SESSION['user_id'])) {
            logActivity('user_logout', 'users', $_SESSION['user_id']);
        }
        session_unset();
        session_destroy();
        session_start();
    }
    
    public function isLoggedIn() {
        return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
    }
    
    public function getUser() {
        if (!$this->isLoggedIn()) return null;
        return $this->getUserById($_SESSION['user_id']);
    }
    
    public function hasPermission($permissionName) {
        return in_array($permissionName, $this->userPermissions);
    }

    private function loadUserPermissions($userId) {
        $user = $this->getUserById($userId);
        if (!$user || !$user['role_id']) {
            $this->userPermissions = [];
            return;
        }
        $sql = "SELECT p.name FROM role_permissions rp JOIN permissions p ON rp.permission_id = p.id WHERE rp.role_id = ?";
        $stmt = $this->db->query($sql, [$user['role_id']]);
        $this->userPermissions = $stmt->fetchAll(PDO::FETCH_COLUMN);
        $_SESSION['permissions'] = $this->userPermissions;
    }
    
    private function createSession($user) {
        session_regenerate_id(true);
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['user_type'] = $user['user_type'];
        $_SESSION['document_number'] = $user['document_number'];
        $_SESSION['email'] = $user['email'];
        $_SESSION['role_id'] = $user['role_id'];
        $_SESSION['logged_in'] = true;
        $_SESSION['login_time'] = time();
        $this->loadUserPermissions($user['id']);
    }

    private function getUserByDocument($documentNumber) {
        $sanitizedNumber = strtoupper(preg_replace('/[^a-zA-Z0-9]/', '', $documentNumber));
        $stmt = $this->db->query("SELECT * FROM users WHERE document_number = ?", [$sanitizedNumber]);
        return $stmt->fetch();
    }
    
    private function getUserById($userId) {
        $stmt = $this->db->query("SELECT * FROM users WHERE id = ?", [$userId]);
        return $stmt->fetch();
    }
    
    private function userExists($documentNumber) {
        return $this->getUserByDocument($documentNumber) !== false;
    }

    private function validateRegistrationData($data) {
        $required = ['user_type', 'document_type', 'document_number', 'email', 'password'];
        foreach ($required as $field) { if (empty($data[$field])) return false; }
        if (!validateEmail($data['email'])) return false;
        if (!validateDocument($data['document_type'], $data['document_number'])) return false;
        if (strlen($data['password']) < 8) return false;
        return true;
    }
    
    public function changePassword($userId, $currentPassword, $newPassword) {
        $user = $this->getUserById($userId);
        if (!$user || !password_verify($currentPassword, $user['password'])) {
            return ['success' => false, 'message' => 'Contraseña actual incorrecta'];
        }
        if (strlen($newPassword) < 8) {
            return ['success' => false, 'message' => 'La nueva contraseña debe tener al menos 8 caracteres'];
        }
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        $stmt = $this->db->query("UPDATE users SET password = ?, updated_at = NOW() WHERE id = ?", [$hashedPassword, $userId]);
        if ($stmt) {
            logActivity('password_changed', 'users', $userId);
            return ['success' => true, 'message' => 'Contraseña cambiada exitosamente'];
        }
        return ['success' => false, 'message' => 'Error al cambiar contraseña'];
    }
    
    public function updateProfile($userId, $data) {
        $allowedFields = ['first_name', 'last_name', 'company_name', 'phone', 'address'];
        $updateFields = []; $params = [];
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $updateFields[] = "$field = ?";
                $params[] = $data[$field];
            }
        }
        if (empty($updateFields)) {
            return ['success' => false, 'message' => 'No hay datos para actualizar'];
        }
        $params[] = $userId;
        $query = "UPDATE users SET " . implode(', ', $updateFields) . ", updated_at = NOW() WHERE id = ?";
        $stmt = $this->db->query($query, $params);
        if ($stmt) {
            logActivity('profile_updated', 'users', $userId);
            return ['success' => true, 'message' => 'Perfil actualizado exitosamente'];
        }
        return ['success' => false, 'message' => 'Error al actualizar perfil'];
    }

    public function resetPassword($email) { /* Tu código original */ }
    public function validateResetToken($token) { /* Tu código original */ }
    public function completePasswordReset($token, $newPassword) { /* Tu código original */ }
}

$GLOBALS['auth'] = new Auth();

function requireAuth() {
    global $auth;
    if (!$auth->isLoggedIn()) {
        header('Location: ' . BASE_URL . 'public/login.php');
        exit;
    }
}

function requireAdmin() {
    global $auth;
    requireAuth();
    if (!$auth->hasPermission('admin_panel_access')) {
        setAlert('Acceso denegado. No tiene permisos para acceder a esta área.', 'error');
        header('Location: ' . BASE_URL . 'public/dashboard.php');
        exit;
    }
}