<?php
// includes/currency.php
class Currency {
    private $db;
    private $defaultCurrency;
    private $currencies = [];
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->loadCurrencies();
    }
    
    private function loadCurrencies() {
        $stmt = $this->db->query("SELECT * FROM currencies WHERE is_active = 1");
        while ($row = $stmt->fetch()) {
            $this->currencies[$row['code']] = $row;
            if ($row['is_default']) {
                $this->defaultCurrency = $row;
            }
        }
    }
    
    public function convert($amount, $fromCurrency, $toCurrency) {
        if ($fromCurrency == $toCurrency) {
            return $amount;
        }
        
        $from = $this->getCurrency($fromCurrency);
        $to = $this->getCurrency($toCurrency);
        
        if (!$from || !$to) {
            return false;
        }
        
        // Convertir a moneda base primero
        $baseAmount = $amount / $from['exchange_rate'];
        // Luego convertir a moneda destino
        return $baseAmount * $to['exchange_rate'];
    }
    
    public function format($amount, $currencyCode = null) {
        if (!$currencyCode) {
            $currency = $this->defaultCurrency;
        } else {
            $currency = $this->getCurrency($currencyCode);
        }
        
        if (!$currency) {
            return number_format($amount, 2);
        }
        
        return $currency['symbol'] . ' ' . number_format($amount, 2, ',', '.');
    }
    
    public function getCurrency($code) {
        return isset($this->currencies[$code]) ? $this->currencies[$code] : null;
    }
    
    public function getAllCurrencies() {
        return $this->currencies;
    }
    
    public function getDefaultCurrency() {
        return $this->defaultCurrency;
    }
    
    public function updateExchangeRate($currencyCode, $newRate) {
        $stmt = $this->db->query(
            "UPDATE currencies SET exchange_rate = ?, last_update = NOW() WHERE code = ?",
            [$newRate, $currencyCode]
        );
        
        if ($stmt) {
            $this->loadCurrencies(); // Recargar monedas
            return true;
        }
        return false;
    }
    
    public function addCurrency($data) {
        $stmt = $this->db->query(
            "INSERT INTO currencies (code, name, symbol, exchange_rate, is_active) VALUES (?, ?, ?, ?, ?)",
            [
                $data['code'],
                $data['name'],
                $data['symbol'],
                $data['exchange_rate'],
                $data['is_active'] ?? 1
            ]
        );
        
        if ($stmt) {
            $this->loadCurrencies();
            return $this->db->lastInsertId();
        }
        return false;
    }
    
    public function setDefaultCurrency($currencyCode) {
        // Primero quitar el default actual
        $this->db->query("UPDATE currencies SET is_default = 0");
        
        // Establecer nuevo default
        $stmt = $this->db->query(
            "UPDATE currencies SET is_default = 1 WHERE code = ?",
            [$currencyCode]
        );
        
        if ($stmt) {
            $this->loadCurrencies();
            return true;
        }
        return false;
    }
    
    // Actualización automática desde BCV (simulado)
    public function updateFromBCV() {
        // Aquí iría la integración real con el API del BCV
        // Por ahora simulamos con valores de ejemplo
        
        $rates = [
            'USD' => 36.50,
            'EUR' => 39.80
        ];
        
        foreach ($rates as $code => $rate) {
            $this->updateExchangeRate($code, $rate);
        }
        
        return true;
    }
}

// Instancia global
$GLOBALS['currency'] = new Currency();
?>
