<?php
// includes/functions.php

function sanitize($input) {
    if (is_array($input)) {
        return array_map('sanitize', $input);
    }
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

function validateDocument($type, $number) {
    $number = preg_replace('/[^0-9A-Za-z]/', '', $number);
    
    switch($type) {
        case 'cedula':
            return preg_match('/^[VE][0-9]{7,8}$/', $number);
        case 'rif':
            return preg_match('/^[JGVEP][0-9]{8,9}$/', $number);
        case 'passport':
            return strlen($number) >= 6;
        default:
            return false;
    }
}

function generateToken($length = 32) {
    return bin2hex(random_bytes($length));
}

function uploadFile($file, $directory = 'general') {
    $uploadDir = UPLOAD_PATH . $directory . '/';
    
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0777, true);
    }
    
    $fileName = uniqid() . '_' . basename($file['name']);
    $targetPath = $uploadDir . $fileName;
    
    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'application/pdf'];
    
    if (!in_array($file['type'], $allowedTypes)) {
        return ['success' => false, 'message' => 'Tipo de archivo no permitido'];
    }
    
    if ($file['size'] > 5000000) { // 5MB
        return ['success' => false, 'message' => 'Archivo demasiado grande'];
    }
    
    if (move_uploaded_file($file['tmp_name'], $targetPath)) {
        return [
            'success' => true, 
            'filename' => $fileName,
            'path' => $directory . '/' . $fileName
        ];
    }
    
    return ['success' => false, 'message' => 'Error al subir archivo'];
}

function sendEmail($to, $subject, $body, $isHTML = true) {
    $headers = "From: " . getSystemConfig('site_email') . "\r\n";
    $headers .= "Reply-To: " . getSystemConfig('site_email') . "\r\n";
    
    if ($isHTML) {
        $headers .= "MIME-Version: 1.0\r\n";
        $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
    }
    
    return mail($to, $subject, $body, $headers);
}

function calculateTaxInterest($amount, $dueDate) {
    $today = new DateTime();
    $due = new DateTime($dueDate);
    
    if ($today <= $due) {
        return 0;
    }
    
    $interval = $today->diff($due);
    $days = $interval->days;
    
    $interestRate = getSystemConfig('tax_interest_rate') ?: 3;
    $monthlyRate = $interestRate / 100;
    
    $months = floor($days / 30);
    $interest = $amount * $monthlyRate * $months;
    
    return round($interest, 2);
}

function generateInvoiceNumber() {
    $year = date('Y');
    $month = date('m');
    
    $db = Database::getInstance();
    $stmt = $db->query(
        "SELECT COUNT(*) as count FROM payments WHERE YEAR(created_at) = ? AND MONTH(created_at) = ?",
        [$year, $month]
    );
    
    $result = $stmt->fetch();
    $sequential = str_pad($result['count'] + 1, 6, '0', STR_PAD_LEFT);
    
    return "INV-{$year}{$month}-{$sequential}";
}

function logActivity($action, $entityType = null, $entityId = null, $oldValues = null, $newValues = null) {
    $db = Database::getInstance();
    
    $userId = $_SESSION['user_id'] ?? null;
    $ipAddress = $_SERVER['REMOTE_ADDR'] ?? '';
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    
    $stmt = $db->query(
        "INSERT INTO audit_logs (user_id, action, entity_type, entity_id, old_values, new_values, ip_address, user_agent) 
         VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
        [
            $userId,
            $action,
            $entityType,
            $entityId,
            $oldValues ? json_encode($oldValues) : null,
            $newValues ? json_encode($newValues) : null,
            $ipAddress,
            $userAgent
        ]
    );
    
    return $stmt ? true : false;
}

function getAlert() {
    if (isset($_SESSION['alert'])) {
        $alert = $_SESSION['alert'];
        unset($_SESSION['alert']);
        return $alert;
    }
    return null;
}

function setAlert($message, $type = 'info') {
    $_SESSION['alert'] = [
        'message' => $message,
        'type' => $type
    ];
}

function formatDate($date, $format = 'd/m/Y') {
    if (!$date) return '';
    return date($format, strtotime($date));
}

function formatDateTime($datetime, $format = 'd/m/Y H:i') {
    if (!$datetime) return '';
    return date($format, strtotime($datetime));
}

function paginate($query, $params = [], $perPage = 20) {
    $db = Database::getInstance();
    
    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $offset = ($page - 1) * $perPage;
    
    // Contar total de registros
    $countQuery = "SELECT COUNT(*) as total FROM (" . $query . ") as count_table";
    $stmt = $db->query($countQuery, $params);
    $total = $stmt->fetch()['total'];
    
    // Obtener registros de la página actual
    $query .= " LIMIT {$perPage} OFFSET {$offset}";
    $stmt = $db->query($query, $params);
    
    return [
        'data' => $stmt->fetchAll(),
        'total' => $total,
        'per_page' => $perPage,
        'current_page' => $page,
        'total_pages' => ceil($total / $perPage)
    ];
}

function generatePaginationLinks($pagination, $baseUrl = '') {
    if ($pagination['total_pages'] <= 1) return '';
    
    $html = '<nav><ul class="pagination">';
    
    // Botón anterior
    if ($pagination['current_page'] > 1) {
        $prevPage = $pagination['current_page'] - 1;
        $html .= "<li class='page-item'><a class='page-link' href='{$baseUrl}?page={$prevPage}'>Anterior</a></li>";
    }
    
    // Páginas
    for ($i = 1; $i <= $pagination['total_pages']; $i++) {
        $active = $i == $pagination['current_page'] ? 'active' : '';
        $html .= "<li class='page-item {$active}'><a class='page-link' href='{$baseUrl}?page={$i}'>{$i}</a></li>";
    }
    
    // Botón siguiente
    if ($pagination['current_page'] < $pagination['total_pages']) {
        $nextPage = $pagination['current_page'] + 1;
        $html .= "<li class='page-item'><a class='page-link' href='{$baseUrl}?page={$nextPage}'>Siguiente</a></li>";
    }
    
    $html .= '</ul></nav>';
    
    return $html;
}
// ... (código existente de functions.php) ...

/**
 * Crea una nueva notificación para un usuario.
 * @param int $userId El ID del usuario que recibirá la notificación.
 * @param string $message El mensaje de la notificación.
 * @param string|null $link Un enlace opcional para la notificación.
 * @return bool True si se creó con éxito, false si no.
 */
function create_notification($userId, $message, $link = null) {
    $db = Database::getInstance();
    $sql = "INSERT INTO notifications (user_id, message, link) VALUES (?, ?, ?)";
    $stmt = $db->query($sql, [$userId, $message, $link]);
    return $stmt !== false;
}
?>

