<?php
class AdvertisementController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Obtiene todos los anuncios registrados por un usuario específico.
     * @param int $userId El ID del usuario.
     * @return array Lista de sus anuncios.
     */
    public function getAdvertisementsByUserId($userId) {
        $sql = "SELECT * FROM advertisements WHERE user_id = ? ORDER BY created_at DESC";
        $stmt = $this->db->query($sql, [$userId]);
        return $stmt === false ? [] : $stmt->fetchAll();
    }

    /**
     * Procesa la solicitud de registro de un nuevo anuncio para verificación.
     * @param int $userId El ID del usuario que registra.
     * @param array $data Los datos del formulario ($_POST).
     * @param array $file El archivo adjunto ($_FILES).
     * @return array Resultado de la operación ['success' => bool, 'message' => string].
     */
    public function requestAdvertisementVerification($userId, $data, $file) {
        // 1. Validar subida del documento
        if (!isset($file['documento_anuncio']) || $file['documento_anuncio']['error'] !== UPLOAD_ERR_OK) {
            return ['success' => false, 'message' => 'Error al subir la foto o permiso.'];
        }

        // 2. Insertar el anuncio en la base de datos
        $sqlAd = "INSERT INTO advertisements (user_id, ad_type, description, location, dimensions, status)
                  VALUES (?, ?, ?, ?, ?, 'pending_verification')";
        
        $paramsAd = [
            $userId,
            $data['ad_type'],
            $data['description'],
            $data['location'],
            $data['dimensions']
        ];
        $stmtAd = $this->db->query($sqlAd, $paramsAd);
        if (!$stmtAd) {
            return ['success' => false, 'message' => 'Error al registrar los datos del anuncio.'];
        }
        $adId = $this->db->lastInsertId();

        // 3. Subir el documento
        $uploadResult = uploadFile($file['documento_anuncio'], 'advertisement_documents');
        if (!$uploadResult['success']) {
            return ['success' => false, 'message' => 'Error al guardar el documento: ' . $uploadResult['message']];
        }

        // 4. Guardar registro del adjunto
        $sqlAttachment = "INSERT INTO attachments (entity_type, entity_id, file_name, file_path, file_type, file_size, uploaded_by)
                          VALUES ('advertisement', ?, ?, ?, ?, ?, ?)";
        $paramsAttachment = [
            $adId, $file['documento_anuncio']['name'], $uploadResult['path'], 
            $file['documento_anuncio']['type'], $file['documento_anuncio']['size'], $userId
        ];
        $stmtAttachment = $this->db->query($sqlAttachment, $paramsAttachment);

        if ($stmtAttachment) {
            logActivity('ad_registration_sent', 'advertisements', $adId);
            return ['success' => true, 'message' => '¡Anuncio registrado! Será revisado por un gestor y se le notificará cuando sea aprobado.'];
        } else {
            // Opcional: Borrar el registro del anuncio si falla el adjunto
            $this->db->query("DELETE FROM advertisements WHERE id = ?", [$adId]);
            return ['success' => false, 'message' => 'El anuncio fue registrado pero hubo un error al guardar el documento.'];
        }
    }
}
?>