<?php
class CitizenTaxController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Obtiene todos los impuestos de un usuario específico.
     * @param int $userId El ID del usuario que ha iniciado sesión.
     * @return array Lista de sus impuestos.
     */
    public function getTaxesByUserId($userId) {
        $sql = "SELECT 
                    t.id, t.amount, t.status, t.due_date,
                    c.code AS currency_code, c.symbol AS currency_symbol,
                    tc.name AS tax_type, t.description
                FROM taxes t
                JOIN currencies c ON t.currency_id = c.id
                JOIN tax_concepts tc ON t.tax_concept_id = tc.id
                WHERE t.user_id = ?
                ORDER BY t.due_date DESC";
        
        $stmt = $this->db->query($sql, [$userId]);
        return $stmt === false ? [] : $stmt->fetchAll();
    }
    
    /**
     * Obtiene todos los vehículos registrados por un usuario específico.
     * @param int $userId El ID del usuario.
     * @return array Lista de sus vehículos.
     */
    public function getVehiclesByUserId($userId) {
        $sql = "SELECT id, plate_number, brand, model, year, status, rejection_reason 
                FROM vehicles 
                WHERE user_id = ? 
                ORDER BY created_at DESC";
        
        $stmt = $this->db->query($sql, [$userId]);
        return $stmt === false ? [] : $stmt->fetchAll();
    }

    /**
     * Procesa la solicitud de registro de un nuevo vehículo para verificación.
     * @param int $userId El ID del usuario que declara.
     * @param array $data Los datos del vehículo desde el formulario ($_POST).
     * @param array $file Los datos del documento adjunto ($_FILES).
     * @return array Un array con el resultado ['success' => bool, 'message' => string].
     */
    public function requestVehicleVerification($userId, $data, $file) {
        $plateNumber = strtoupper(trim($data['plate_number']));

        // 1. Verificar si la placa ya existe en el sistema.
        $stmt = $this->db->query("SELECT id, user_id FROM vehicles WHERE plate_number = ?", [$plateNumber]);
        $existingVehicle = $stmt->fetch();

        if ($existingVehicle) {
            if ($existingVehicle['user_id'] != $userId) {
                return ['success' => false, 'message' => 'Error: La placa ' . htmlspecialchars($plateNumber) . ' ya está registrada por otro contribuyente.'];
            }
        }

        // 2. Validar que el archivo se haya subido correctamente.
        if (!isset($file['documento_vehiculo']) || $file['documento_vehiculo']['error'] !== UPLOAD_ERR_OK) {
            return ['success' => false, 'message' => 'Error al subir el documento. Por favor, inténtelo de nuevo.'];
        }

        // 3. Insertar o Actualizar el vehículo en la tabla 'vehicles' con estado 'pending_verification'.
        $sqlVehicle = "INSERT INTO vehicles (user_id, plate_number, brand, model, year, vehicle_type, status, rejection_reason) 
                       VALUES (?, ?, ?, ?, ?, ?, 'pending_verification', NULL)
                       ON DUPLICATE KEY UPDATE brand=VALUES(brand), model=VALUES(model), year=VALUES(year), vehicle_type=VALUES(vehicle_type), status='pending_verification', rejection_reason=NULL";
        
        $paramsVehicle = [
            $userId, $plateNumber, $data['brand'], $data['model'], $data['year'], $data['vehicle_type']
        ];

        $stmtVehicle = $this->db->query($sqlVehicle, $paramsVehicle);
        if (!$stmtVehicle) {
            return ['success' => false, 'message' => 'Error al registrar los datos del vehículo.'];
        }

        $vehicleId = $this->db->lastInsertId();
        if ($vehicleId == 0) {
            $vehicleId = $existingVehicle['id'];
        }

        // 4. Subir el archivo adjunto.
        $uploadResult = uploadFile($file['documento_vehiculo'], 'vehicle_documents');
        if (!$uploadResult['success']) {
            return ['success' => false, 'message' => 'Error al guardar el documento: ' . $uploadResult['message']];
        }

        // 5. Guardar la información del archivo en la tabla 'attachments'.
        // Opcional: Borrar adjuntos viejos del mismo vehículo para mantener solo el más reciente.
        $this->db->query("DELETE FROM attachments WHERE entity_type = 'vehicle' AND entity_id = ?", [$vehicleId]);
        
        $sqlAttachment = "INSERT INTO attachments (entity_type, entity_id, file_name, file_path, file_type, file_size, uploaded_by)
                          VALUES ('vehicle', ?, ?, ?, ?, ?, ?)";
        
        $paramsAttachment = [
            $vehicleId, $file['documento_vehiculo']['name'], $uploadResult['path'], $file['documento_vehiculo']['type'], $file['documento_vehiculo']['size'], $userId
        ];

        $stmtAttachment = $this->db->query($sqlAttachment, $paramsAttachment);

        if ($stmtAttachment) {
            logActivity('vehicle_declaration_sent', 'vehicles', $vehicleId);
            return ['success' => true, 'message' => '¡Declaración enviada! Su vehículo será revisado por un gestor y será notificado cuando sea aprobado.'];
        } else {
            return ['success' => false, 'message' => 'El vehículo fue registrado pero hubo un error al guardar el documento adjunto.'];
        }
    }

    /**
     * Obtiene todas las actividades económicas registradas por un usuario.
     * @param int $userId El ID del usuario.
     * @return array Lista de sus actividades económicas.
     */
    public function getEconomicActivitiesByUserId($userId) {
        $sql = "SELECT * FROM economic_activities WHERE user_id = ? ORDER BY created_at DESC";
        $stmt = $this->db->query($sql, [$userId]);
        return $stmt === false ? [] : $stmt->fetchAll();
    }
    
    /**
     * Procesa la declaración de ingresos brutos y genera el impuesto ISAE.
     * @param int $userId El ID del usuario.
     * @param array $data Datos del formulario de declaración.
     * @return array Resultado de la operación.
     */
    public function declareIsae($userId, $data) {
        // 1. Validar datos de entrada
        $activityId = $data['activity_id'];
        $grossIncome = floatval($data['gross_income']);
        $month = intval($data['declaration_period_month']);
        $year = intval($data['declaration_period_year']);

        // 2. Obtener el concepto tributario para ISAE desde la configuración
        $conceptId = getSystemConfig('isae_tax_concept_id');
        if (empty($conceptId)) {
            return ['success' => false, 'message' => 'El concepto para ISAE no está configurado por el administrador.'];
        }
        $concept = $this->db->query("SELECT * FROM tax_concepts WHERE id = ?", [$conceptId])->fetch();
        if (!$concept || empty($concept['aliquota'])) {
            return ['success' => false, 'message' => 'El concepto para ISAE configurado es inválido o no tiene una alícuota definida.'];
        }

        // 3. Verificar que la actividad económica pertenece al usuario y está activa
        $activity = $this->db->query("SELECT * FROM economic_activities WHERE id = ? AND user_id = ? AND status = 'active'", [$activityId, $userId])->fetch();
        if (!$activity) {
            return ['success' => false, 'message' => 'La actividad económica seleccionada no es válida o no está activa.'];
        }

        // 4. Calcular el impuesto
        $aliquota = floatval($concept['aliquota']);
        $taxAmount = $grossIncome * ($aliquota / 100);
        $minimumTax = floatval($concept['base_amount'] ?? 0);

        if ($taxAmount < $minimumTax) {
            $taxAmount = $minimumTax;
        }

        // 5. Crear la deuda en la tabla 'taxes'
        $defaultCurrency = $this->db->query("SELECT id FROM currencies WHERE is_default = 1")->fetch();
        $description = "Declaración ISAE - " . $activity['activity_code'] . " - Período: " . $month . "/" . $year;
        $dueDate = date('Y-m-d', strtotime('+15 days'));

        $sqlTax = "INSERT INTO taxes (user_id, tax_concept_id, description, amount, currency_id, due_date, status) VALUES (?, ?, ?, ?, ?, ?, 'pending')";
        $paramsTax = [$userId, $conceptId, $description, round($taxAmount, 2), $defaultCurrency['id'], $dueDate];
        $stmtTax = $this->db->query($sqlTax, $paramsTax);
        if (!$stmtTax) {
            return ['success' => false, 'message' => 'Error al generar la deuda del impuesto.'];
        }
        $taxId = $this->db->lastInsertId();
        
        // 6. Guardar el registro en el historial de declaraciones ISAE
        $sqlHistory = "INSERT INTO isae_declarations (economic_activity_id, tax_id, declaration_period_month, declaration_period_year, gross_income, tax_amount, aliquota_applied) VALUES (?, ?, ?, ?, ?, ?, ?)";
        $paramsHistory = [$activityId, $taxId, $month, $year, $grossIncome, round($taxAmount, 2), $aliquota];
        $this->db->query($sqlHistory, $paramsHistory);

        logActivity('isae_declared', 'isae_declarations', $this->db->lastInsertId());
        return ['success' => true, 'message' => 'Declaración de ISAE procesada exitosamente. Su deuda ha sido generada.'];
    }
}
?>