<?php
class LicenseController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Obtiene todas las licencias y solicitudes de un usuario.
     */
    public function getLicensesByUserId($userId) {
        $sql = "SELECT l.*, e.description as activity_description
                FROM economic_activity_licenses l
                JOIN economic_activities e ON l.economic_activity_id = e.id
                WHERE l.user_id = ? 
                ORDER BY l.request_date DESC";
        $stmt = $this->db->query($sql, [$userId]);
        return $stmt->fetchAll();
    }

    /**
     * Obtiene las actividades económicas de un usuario que están activas
     * y que no tienen una licencia ya solicitada o activa.
     */
    public function getEligibleActivitiesForLicense($userId) {
        $sql = "SELECT * FROM economic_activities 
                WHERE user_id = ? AND status = 'active'
                AND id NOT IN (
                    SELECT economic_activity_id FROM economic_activity_licenses 
                    WHERE user_id = ? AND status IN ('solicitada', 'en_revision', 'aprobada')
                )";
        $stmt = $this->db->query($sql, [$userId, $userId]);
        return $stmt->fetchAll();
    }

    /**
     * Crea una nueva solicitud de licencia.
     */
    public function createLicenseRequest($userId, $activityId) {
        // Validar que la actividad sea elegible
        $eligibleActivities = $this->getEligibleActivitiesForLicense($userId);
        $isEligible = false;
        foreach ($eligibleActivities as $activity) {
            if ($activity['id'] == $activityId) {
                $isEligible = true;
                break;
            }
        }

        if (!$isEligible) {
            return ['success' => false, 'message' => 'Esta actividad económica no es elegible para una solicitud de licencia en este momento.'];
        }

        $sql = "INSERT INTO economic_activity_licenses (user_id, economic_activity_id, status) VALUES (?, ?, 'solicitada')";
        $stmt = $this->db->query($sql, [$userId, $activityId]);
        
        if ($stmt) {
            logActivity('license_requested', 'economic_activity_licenses', $this->db->lastInsertId());
            return ['success' => true, 'message' => 'Solicitud de licencia enviada exitosamente. Será revisada por un administrador.'];
        }

        return ['success' => false, 'message' => 'Error al procesar la solicitud.'];
    }
}
?>