<?php
class PaymentController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Procesa el reporte de un pago de un impuesto.
     * @param int $userId ID del usuario que reporta.
     * @param array $data Datos del formulario ($_POST).
     * @param array $file Datos del archivo subido ($_FILES).
     * @return array Un array con el resultado ['success' => bool, 'message' => string].
     */
    public function reportPayment($userId, $data, $file) {
        // 1. Validar que el impuesto pertenezca al usuario y esté pendiente de pago
        $taxStmt = $this->db->query(
            "SELECT id, amount, currency_id FROM taxes WHERE id = ? AND user_id = ? AND status IN ('pending', 'overdue')",
            [$data['tax_id'], $userId]
        );
        $tax = $taxStmt->fetch();

        if (!$tax) {
            return ['success' => false, 'message' => 'El impuesto no es válido o ya tiene un pago en proceso.'];
        }

        // 2. Gestionar la subida del comprobante (si existe)
        $receiptPath = null;
        if (isset($file['receipt']) && $file['receipt']['error'] === UPLOAD_ERR_OK) {
            // Usamos la función que ya existe en includes/functions.php
            $uploadResult = uploadFile($file['receipt'], 'receipts'); 
            if ($uploadResult['success']) {
                $receiptPath = $uploadResult['path'];
            } else {
                return ['success' => false, 'message' => 'Error al subir el comprobante: ' . $uploadResult['message']];
            }
        }

        // 3. Insertar el registro del pago en la tabla 'payments'
        $sql = "INSERT INTO payments (user_id, tax_id, amount, currency_id, payment_method, reference_number, bank_name, status, receipt_url, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, 'pending', ?, ?)";
        
        $params = [
            $userId,
            $data['tax_id'],
            $tax['amount'], // Usamos el monto de la deuda para seguridad
            $tax['currency_id'],
            $data['payment_method'],
            $data['reference_number'],
            $data['bank_name'],
            $receiptPath,
            $data['payment_date'] // Asumimos que el campo en BD es `created_at` pero usamos la fecha del pago
        ];

        $paymentStmt = $this->db->query($sql, $params);

        if (!$paymentStmt) {
            return ['success' => false, 'message' => 'Ocurrió un error al registrar el pago.'];
        }

        // 4. Actualizar el estado del impuesto a 'in_verification'
        $updateTaxStmt = $this->db->query(
            "UPDATE taxes SET status = 'in_verification' WHERE id = ?",
            [$data['tax_id']]
        );
        
        if ($updateTaxStmt) {
            logActivity('payment_reported', 'payments', $this->db->lastInsertId());
            return ['success' => true, 'message' => '¡Pago notificado exitosamente! Será verificado por un administrador a la brevedad.'];
        }

        return ['success' => false, 'message' => 'El pago se registró, pero no se pudo actualizar el estado del impuesto.'];
    }
}
?>