<?php
class PermitController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Obtiene todos los permisos solicitados por un usuario.
     */
    public function getPermitsByUserId($userId) {
        $sql = "SELECT pa.*, pt.name as permit_name
                FROM permit_applications pa
                JOIN permit_types pt ON pa.permit_type_id = pt.id
                WHERE pa.user_id = ? 
                ORDER BY pa.application_date DESC";
        $stmt = $this->db->query($sql, [$userId]);
        return $stmt->fetchAll();
    }

    /**
     * Obtiene los tipos de permisos de construcción disponibles.
     */
    public function getConstructionPermitTypes() {
        $sql = "SELECT * FROM permit_types WHERE issuing_department = 'Dirección de Urbanismo'";
        return $this->db->query($sql)->fetchAll();
    }

    /**
     * Obtiene los inmuebles verificados de un usuario para asociarlos a un permiso.
     */
    public function getVerifiedPropertiesByUserId($userId) {
        $sql = "SELECT id, cadastral_code, address FROM properties WHERE user_id = ? AND status = 'verified'";
        return $this->db->query($sql, [$userId])->fetchAll();
    }

    /**
     * Crea una nueva solicitud de permiso de construcción.
     */
    public function createConstructionPermitRequest($userId, $data, $file) {
        // Validación de datos
        if (empty($data['permit_type_id']) || empty($data['property_id'])) {
            return ['success' => false, 'message' => 'Debe seleccionar el tipo de permiso y el inmueble asociado.'];
        }
        if (!isset($file['document_support']) || $file['document_support']['error'] !== UPLOAD_ERR_OK) {
            return ['success' => false, 'message' => 'Es obligatorio adjuntar los planos o documentos de soporte.'];
        }

        // Subir el archivo adjunto
        $uploadResult = uploadFile($file['document_support'], 'permit_documents');
        if (!$uploadResult['success']) {
            return ['success' => false, 'message' => 'Error al guardar el documento: ' . $uploadResult['message']];
        }

        // Insertar la solicitud de permiso
        $sql = "INSERT INTO permit_applications (user_id, permit_type_id, related_entity_type, related_entity_id, notes, status) 
                VALUES (?, ?, 'property', ?, ?, 'solicitado')";
        $params = [$userId, $data['permit_type_id'], $data['property_id'], $data['notes'] ?? null];
        $stmt = $this->db->query($sql, $params);

        if (!$stmt) {
            return ['success' => false, 'message' => 'Error al registrar la solicitud de permiso.'];
        }
        $permitId = $this->db->lastInsertId();

        // Guardar el registro del adjunto
        $sqlAttachment = "INSERT INTO attachments (entity_type, entity_id, file_name, file_path, file_type, file_size, uploaded_by)
                          VALUES ('permit_application', ?, ?, ?, ?, ?, ?)";
        $paramsAttachment = [
            $permitId, $file['document_support']['name'], $uploadResult['path'], 
            $file['document_support']['type'], $file['document_support']['size'], $userId
        ];
        $this->db->query($sqlAttachment, $paramsAttachment);

        logActivity('construction_permit_requested', 'permit_applications', $permitId);
        return ['success' => true, 'message' => '¡Solicitud de permiso enviada! Será revisada por la Dirección de Urbanismo.'];
    }

    // --- INICIO DE NUEVAS FUNCIONES ---

    /**
     * Obtiene los tipos de permisos para eventos y economía informal.
     */
    public function getEventPermitTypes() {
        $sql = "SELECT * FROM permit_types WHERE issuing_department IN ('Dirección de Espectáculos Públicos', 'Dirección de Hacienda')";
        return $this->db->query($sql)->fetchAll();
    }

    /**
     * Crea una nueva solicitud de permiso para eventos o economía informal.
     */
    public function createEventPermitRequest($userId, $data, $file) {
        // Validación de datos
        if (empty($data['permit_type_id'])) {
            return ['success' => false, 'message' => 'Debe seleccionar el tipo de permiso.'];
        }
        if (!isset($file['document_support']) || $file['document_support']['error'] !== UPLOAD_ERR_OK) {
            return ['success' => false, 'message' => 'Es obligatorio adjuntar un documento de soporte (Ej: cédula, RIF, etc.).'];
        }

        // Subir el archivo adjunto
        $uploadResult = uploadFile($file['document_support'], 'permit_documents');
        if (!$uploadResult['success']) {
            return ['success' => false, 'message' => 'Error al guardar el documento: ' . $uploadResult['message']];
        }

        // Insertar la solicitud de permiso
        $sql = "INSERT INTO permit_applications (user_id, permit_type_id, notes, status) VALUES (?, ?, ?, 'solicitado')";
        $params = [$userId, $data['permit_type_id'], $data['notes'] ?? null];
        $stmt = $this->db->query($sql, $params);

        if (!$stmt) {
            return ['success' => false, 'message' => 'Error al registrar la solicitud de permiso.'];
        }
        $permitId = $this->db->lastInsertId();

        // Guardar el registro del adjunto
        $sqlAttachment = "INSERT INTO attachments (entity_type, entity_id, file_name, file_path, file_type, file_size, uploaded_by)
                          VALUES ('permit_application', ?, ?, ?, ?, ?, ?)";
        $paramsAttachment = [
            $permitId, $file['document_support']['name'], $uploadResult['path'], 
            $file['document_support']['type'], $file['document_support']['size'], $userId
        ];
        $this->db->query($sqlAttachment, $paramsAttachment);

        logActivity('event_permit_requested', 'permit_applications', $permitId);
        return ['success' => true, 'message' => '¡Solicitud de permiso enviada exitosamente!'];
    }
    // --- FIN DE NUEVAS FUNCIONES ---
}
?>