<?php
class PropertyController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Obtiene todos los inmuebles registrados por un usuario específico.
     * @param int $userId El ID del usuario.
     * @return array Lista de sus inmuebles.
     */
    public function getPropertiesByUserId($userId) {
        $sql = "SELECT * FROM properties WHERE user_id = ? ORDER BY created_at DESC";
        $stmt = $this->db->query($sql, [$userId]);
        return $stmt === false ? [] : $stmt->fetchAll();
    }

    /**
     * Procesa la solicitud de registro de un nuevo inmueble para verificación.
     * @param int $userId El ID del usuario que registra.
     * @param array $data Los datos del formulario ($_POST).
     * @param array $file El archivo adjunto ($_FILES).
     * @return array Resultado de la operación ['success' => bool, 'message' => string].
     */
    public function requestPropertyVerification($userId, $data, $file) {
        $cadastralCode = strtoupper(trim($data['cadastral_code']));

        // 1. Verificar si la cédula catastral ya existe
        $stmt = $this->db->query("SELECT id, user_id FROM properties WHERE cadastral_code = ?", [$cadastralCode]);
        $existingProperty = $stmt->fetch();
        if ($existingProperty && $existingProperty['user_id'] != $userId) {
            return ['success' => false, 'message' => 'Error: La Cédula Catastral ' . htmlspecialchars($cadastralCode) . ' ya está registrada por otro contribuyente.'];
        }

        // 2. Validar subida del documento
        if (!isset($file['documento_inmueble']) || $file['documento_inmueble']['error'] !== UPLOAD_ERR_OK) {
            return ['success' => false, 'message' => 'Error al subir el documento de propiedad.'];
        }

        // 3. Insertar o actualizar el inmueble
        $sqlProperty = "INSERT INTO properties (user_id, cadastral_code, address, area_m2, property_type, status, rejection_reason)
                        VALUES (?, ?, ?, ?, ?, 'pending_verification', NULL)
                        ON DUPLICATE KEY UPDATE address=VALUES(address), area_m2=VALUES(area_m2), property_type=VALUES(property_type), status='pending_verification', rejection_reason=NULL";
        
        $paramsProperty = [$userId, $cadastralCode, $data['address'], $data['area_m2'], $data['property_type']];
        $stmtProperty = $this->db->query($sqlProperty, $paramsProperty);
        if (!$stmtProperty) {
            return ['success' => false, 'message' => 'Error al registrar los datos del inmueble.'];
        }
        $propertyId = $this->db->lastInsertId();
        if ($propertyId == 0) {
            $propertyId = $existingProperty['id'];
        }

        // 4. Subir el documento
        $uploadResult = uploadFile($file['documento_inmueble'], 'property_documents');
        if (!$uploadResult['success']) {
            return ['success' => false, 'message' => 'Error al guardar el documento: ' . $uploadResult['message']];
        }

        // 5. Guardar registro del adjunto
        $this->db->query("DELETE FROM attachments WHERE entity_type = 'property' AND entity_id = ?", [$propertyId]);
        $sqlAttachment = "INSERT INTO attachments (entity_type, entity_id, file_name, file_path, file_type, file_size, uploaded_by)
                          VALUES ('property', ?, ?, ?, ?, ?, ?)";
        $paramsAttachment = [
            $propertyId, $file['documento_inmueble']['name'], $uploadResult['path'], 
            $file['documento_inmueble']['type'], $file['documento_inmueble']['size'], $userId
        ];
        $stmtAttachment = $this->db->query($sqlAttachment, $paramsAttachment);

        if ($stmtAttachment) {
            logActivity('property_registration_sent', 'properties', $propertyId);
            return ['success' => true, 'message' => '¡Inmueble registrado! Será revisado por un gestor y se le notificará cuando sea aprobado.'];
        } else {
            return ['success' => false, 'message' => 'El inmueble fue registrado pero hubo un error al guardar el documento.'];
        }
    }
}
?>