<?php
class ReportController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Obtiene todos los reportes creados por un usuario específico.
     * @param int $userId El ID del usuario.
     * @return array La lista de sus reportes.
     */
    public function getReportsByUserId($userId) {
        $sql = "SELECT * FROM citizen_reports WHERE user_id = ? ORDER BY created_at DESC";
        $stmt = $this->db->query($sql, [$userId]);
        return $stmt === false ? [] : $stmt->fetchAll();
    }

    /**
     * Crea un nuevo reporte ciudadano.
     * @param int $userId El ID del usuario que crea el reporte.
     * @param array $data Los datos del formulario ($_POST).
     * @param array $file El archivo adjunto opcional ($_FILES).
     * @return array Un array con el resultado ['success' => bool, 'message' => string].
     */
    public function createReport($userId, $data, $file) {
        // 1. Insertar los datos básicos del reporte
        $sql = "INSERT INTO citizen_reports (user_id, category, title, description, location) 
                VALUES (?, ?, ?, ?, ?)";
        
        $params = [
            $userId,
            $data['category'],
            $data['title'],
            $data['description'],
            $data['location']
        ];
        
        $stmt = $this->db->query($sql, $params);
        if (!$stmt) {
            return ['success' => false, 'message' => 'Error al guardar el reporte en la base de datos.'];
        }
        $reportId = $this->db->lastInsertId();

        // 2. Si se adjuntó una foto, procesarla y guardarla
        if (isset($file['report_file']) && $file['report_file']['error'] === UPLOAD_ERR_OK) {
            $uploadResult = uploadFile($file['report_file'], 'citizen_reports');
            if ($uploadResult['success']) {
                $sqlAttachment = "INSERT INTO attachments (entity_type, entity_id, file_name, file_path, file_type, file_size, uploaded_by)
                                  VALUES ('citizen_report', ?, ?, ?, ?, ?, ?)";
                
                $paramsAttachment = [
                    $reportId,
                    $file['report_file']['name'],
                    $uploadResult['path'],
                    $file['report_file']['type'],
                    $file['report_file']['size'],
                    $userId
                ];
                $this->db->query($sqlAttachment, $paramsAttachment);
            } else {
                // Si la subida del archivo falla, el reporte ya se creó, pero se puede notificar el error del adjunto.
                setAlert('Reporte creado, pero hubo un error al subir la foto: ' . $uploadResult['message'], 'warning');
            }
        }

        logActivity('citizen_report_created', 'citizen_reports', $reportId);
        return ['success' => true, 'message' => '¡Reporte enviado exitosamente! Gracias por su colaboración.'];
    }
}
?>