<?php
class SolvencyController {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Verifica si un usuario está solvente (no tiene deudas pendientes o vencidas).
     * @param int $userId El ID del usuario.
     * @return array Un array con el estado de solvencia y la lista de deudas pendientes.
     */
    public function checkSolvencyStatus($userId) {
        $sql = "SELECT t.id, tc.name as concept, t.amount, t.due_date, c.symbol 
                FROM taxes t
                JOIN tax_concepts tc ON t.tax_concept_id = tc.id
                JOIN currencies c ON t.currency_id = c.id
                WHERE t.user_id = ? AND t.status IN ('pending', 'overdue')";
        
        $stmt = $this->db->query($sql, [$userId]);
        $pendingTaxes = $stmt->fetchAll();

        return [
            'is_solvent' => empty($pendingTaxes),
            'pending_taxes' => $pendingTaxes
        ];
    }

    /**
     * Obtiene el historial de solvencias emitidas para un usuario.
     * @param int $userId El ID del usuario.
     * @return array Lista de solvencias.
     */
    public function getSolvencyHistory($userId) {
        $sql = "SELECT * FROM solvencies WHERE user_id = ? ORDER BY issue_date DESC";
        $stmt = $this->db->query($sql, [$userId]);
        return $stmt->fetchAll();
    }

    /**
     * Genera una nueva solvencia para un usuario si está solvente.
     * @param int $userId El ID del usuario.
     * @return array Resultado de la operación.
     */
    public function generateSolvency($userId) {
        // Primero, re-verificamos que el usuario esté solvente por seguridad.
        if (!$this->checkSolvencyStatus($userId)['is_solvent']) {
            return ['success' => false, 'message' => 'No se puede generar la solvencia. Tiene deudas pendientes.'];
        }

        $solvencyType = 'Solvencia Municipal General';
        $uniqueCode = 'SM-' . date('Ymd') . '-' . strtoupper(bin2hex(random_bytes(4)));
        $issueDate = date('Y-m-d');
        $expiryDate = date('Y-m-d', strtotime('+90 days'));

        $sql = "INSERT INTO solvencies (user_id, solvency_type, unique_code, issue_date, expiry_date) VALUES (?, ?, ?, ?, ?)";
        $params = [$userId, $solvencyType, $uniqueCode, $issueDate, $expiryDate];
        $stmt = $this->db->query($sql, $params);

        if ($stmt) {
            logActivity('solvency_generated', 'solvencies', $this->db->lastInsertId());
            return ['success' => true, 'message' => '¡Solvencia generada exitosamente!'];
        }

        return ['success' => false, 'message' => 'Error al guardar la solvencia en la base de datos.'];
    }
}
?>